/** \file ble_cube_lightingPattern.c
 *
 * \brief Cube lighting pattern service is implemented here
 */
 
/* INCLUDE FILES */
#include "CUBE_BLE_lightingPattern.h"
#include "CUBE_IO_BAT.h"
#include "nrf_delay.h"
#include "string.h"
#include "nordic_common.h"
//#include "ble_srv_common.h"
//#include "app_util.h"
#include "CUBE_IO_LED.h"
#include "cmsis_os.h"
#include "CUBE_timers.h"
#include "CUBE_APP_SoundResponse.h"

/* MODULE INTERNAL CONSTANT DEFINITIONS    *ccccccc*/

/* MODULE INTERNAL MACRO DEFINITIONS       *mmmmmmm*/


typedef enum{
BLINK_CMD_INDEX_COMMAND,		//0
BLINK_CMD_INDEX_NUM_OF_BLINKS,	//1
BLINK_CMD_INDEX_DELAY_MS_LSB,	//2
BLINK_CMD_INDEX_DELAY_MS_MSB,	//3
BLINK_CMD_INDEX_RED,			//4
BLINK_CMD_INDEX_GREEN,			//5
BLINK_CMD_INDEX_BLUE,			//6
BLINK_CMD_INDEX_STEP_SIZE,		//7
}blink_cmd_index_t;

typedef enum{
SETRGB_CMD_INDEX_COMMAND,		//0
SETRGB_CMD_INDEX_RED,			//1
SETRGB_CMD_INDEX_GREEN,			//2
SETRGB_CMD_INDEX_BLUE,			//3
SETRGB_CMD_INDEX_STEP_SIZE,		//4
SETRGB_CMD_INDEX_DELAY_LSB,		//4
SETRGB_CMD_INDEX_DELAY_MSB,		//4
}setrgb_cmd_index_t;

typedef enum{
	COMMAND_INDEX,
	MASTER_FLAG_INDEX,
	STEP_SIZE_INDEX
}set_master_cmd_index_t;
/* MODULE INTERNAL TYPE DEFINITIONS        *ttttttt*/
typedef enum{
	LP_CMD_SET_LIGHTING_PATTERN = 1,
	LP_CMD_SET_RGB,
	LP_CMD_BLINK_RGB,
	LP_CMD_RESET_CUBE,
	LP_CMD_SET_MASTER,
	LP_CMD_SET_BASE_COLOR = 8,
	LP_CMD_FW_VERSION  = 9,
 	LP_CMD_FOTA_UPDATE = 10,
	LP_CMD_SOUND_DISABLE = 12,
	LP_CMD_SOUND_ENABLE = 11
}lp_commands_t;

typedef enum{
	CURRENT_PATTERN_INDEX,
	PATTERN_ERR_CODE_INDEX,
	MASTER_STEP_SIZE_INDEX,
	MAX_PATTERN_INFO_INDEX
}pattern_info_index_t;

/* MODULE INTERNAL DATA DEFINITIONS        *ddddddd*/


// The structure which holds all data for the lighting pattern service
static ble_cube_lp_t gs_lightingPatternService;

// Current lighting pattern 
lp_patterns_t ge_currentLightingPattern;

volatile uint8_t gvu8_color[3];
/* MODULE INTERNAL FUNCTION PROTOTYPES     *fffffff*/

// Add characteristic : Cube Data out to lighting pattern service
static uint32_t char_add_cubeDataOut(void);

// Add characteristic : Cube Command to lighting pattern service
static uint32_t char_add_cubeCommand(void);

// Add characteristic : Pattern Info to lighting pattern service
static uint32_t char_add_patternInfo(void);

#if 0
// Add characteristic : Pattern Config to lighting pattern service
static uint32_t char_add_patternConfig(void);
#endif

// Handles the Connect event
static void lp_on_connect(ble_evt_t * p_ble_evt);

// Handles the disconnect event
static void lp_on_disconnect(ble_evt_t * p_ble_evt);

// Handles the Write event, calls the appropriate write handler for the service attribute
static void lp_on_write(ble_evt_t * p_ble_evt);

// Handler for Command attribute write event
static bool lp_command_write_handler(uint8_t* rpu8_newState);

// Handler for pattern config attribute write event
static bool lp_patternConfig_write_handler(uint8_t* rpu8_newState);

// Function to update the current lighting pattern
static void lp_updateLightingPattern(uint8_t ru8_pattern,uint8_t* rpu8_patternInfo);

// Function to handle CCCD write for data out command (enable/disable notifications)
static void lp_cccdWriteDataOutCommand(ble_gatts_evt_write_t * p_evt_write);

#define		FW_VERSION_OPCODE 	0x09
static uint8_t FW_version_response[4]  = {FW_VERSION_OPCODE, 1,4,2}; // Firmware Version

uint8_t OTA_ENABLE = 0;
volatile uint8_t Sound_Enable = 0;
volatile uint8_t Sound_Disable = 0;
volatile uint8_t CUBE_Sound_Pattern = 0;

extern osThreadId CUBE_APP_SoundResponseThread_id;

/* EXPORTED (GLOBAL or External) FUNCTIONS *ggggggg*/
extern  SoundResponseState_t ge_SoundResponseState;

/**@brief Function for adding the Cube common Service
 *
 * @param[in]   p_cube_cmn        Cube common Service structure.
 * @param[in]   p_cube_cmn_init   Information needed to initialize the service.
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 
	@todo 		The lighting pattern service is initialized as BLE_GATTS_SRVC_TYPE_PRIMARY here. Should it be secondary?
 */
uint32_t lp_service_init(void)
{
	uint32_t   		lu32_errCode;
	ble_uuid_t 		ls_ble_uuid;
	ble_uuid128_t 	ls_service_base_uuid = CUBE_LP_SERVICE_UUID_BASE ;

    // Initialize service structure
	
	// Set handle as invalid, will be assigned by softdevice
    gs_lightingPatternService.conn_handle               = BLE_CONN_HANDLE_INVALID;
	
	//	Initialize the event handler. Currently not used, hence set to NULL
	gs_lightingPatternService.evt_handler = NULL;
	
	// Initialize the write handler for command write
	gs_lightingPatternService.mf_cubeCommandWriteHandler = lp_command_write_handler;
	
	// Initialize the write handler for pattern config
	gs_lightingPatternService.mf_patternConfigWriteHandler = lp_patternConfig_write_handler;

	//------------------Add the custom UUID base to the stack------------------
	//	gs_lightingPatternService->uuid_type : will contain the result of the function
	lu32_errCode = sd_ble_uuid_vs_add(&ls_service_base_uuid, & (gs_lightingPatternService.uuid_type));
	if(lu32_errCode != NRF_SUCCESS)
	{
		//	If there was a problem doing this task, exit with error code
		return lu32_errCode;
	}
	
	//	Store the UUID and the UUID type in the UUID structure
	ls_ble_uuid.type = gs_lightingPatternService.uuid_type;
	ls_ble_uuid.uuid = CUBE_LP_UUID_SERVICE;
	
	//	Add this service to the GATT table. Set it as Primary service type. Should it be secondary? 
	lu32_errCode = sd_ble_gatts_service_add(BLE_GATTS_SRVC_TYPE_PRIMARY, &ls_ble_uuid, &gs_lightingPatternService.service_handle);
    if (lu32_errCode != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return lu32_errCode;
    }

    // Add characteristic : Cube Data Out
    lu32_errCode = char_add_cubeDataOut();
    if (lu32_errCode != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return lu32_errCode;
    }
	
	// Disable Cube Data Out notifications
	device_param_setDataOutEnabled(false);
	
	// Set Cube as slave
	device_param_setAsSlave();
	
	// Set step size as default step size
    device_param_setmasterStepSize(DEFAULT_STEP_SIZE);
	
	// Add characteristic : Cube Command
    lu32_errCode = char_add_cubeCommand();
    if (lu32_errCode != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return lu32_errCode;
    }

	// Add characteristic : Pattern Info
    lu32_errCode = char_add_patternInfo();
    if (lu32_errCode != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return lu32_errCode;
    }

#if 0
	// Add characteristic : Pattern Config
    lu32_errCode = char_add_patternConfig();
    if (lu32_errCode != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return lu32_errCode;
    }
#endif
	
	//Initialize current lighting patter
	ge_currentLightingPattern = LP_PATTERN_NONE;
	
    return NRF_SUCCESS;
}

/**@brief 	Function to get the UUID type for the lighting pattern service
 *
 * @details 
 *
 * @param[in]   rpu8_newState  	Write data
 */
uint8_t	lp_getUuidType(void)
{
	return gs_lightingPatternService.uuid_type;
}

/**@brief Function for handling the Application's BLE Stack events.
 *
 * @details Handles all events from the BLE stack of interest to the CUBE Lighting Pattern Service.
 *          This is called from ble_evt_dispatch() in main()
 *
 * @param[in]   p_ble_evt  	Event received from the BLE stack.
 */
void lp_on_ble_evt(ble_evt_t * p_ble_evt)
{
    switch (p_ble_evt->header.evt_id)
    {
        case BLE_GAP_EVT_CONNECTED:
            lp_on_connect(p_ble_evt);
            break;

        case BLE_GAP_EVT_DISCONNECTED:
            lp_on_disconnect(p_ble_evt);
            break;

        case BLE_GATTS_EVT_WRITE:
            lp_on_write(p_ble_evt);
            break;

        default:
            // No implementation needed.
            break;
    }
}

/**@brief Function for notifying the euler angles calculated using MPU
 * @param[in]   p_xyz_values   	euler angles
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
uint32_t lp_notify_dataOut(void)
{
	ble_gatts_hvx_params_t params;					//	This will be used to specify the operations to be done
	uint16_t	len = 3*sizeof(uint8_t);
	uint32_t 	err_code = NRF_SUCCESS;
	static uint8_t lau8_dataToNotify[CUBE_PATTERN_ATTR_SIZE_MAX];
	
	// if notifications are not enabled, return
	if(device_param_getDataOutEnabled() == false)
	{
		return NRF_SUCCESS;
	}
	
	//	get the Euler angles
	//app_accel_getEulerAngles(lau8_dataToNotify);
	//len = 12*sizeof(uint8_t);
	
	// Get RGB data
	getCurrentLedData(lau8_dataToNotify);	
	lau8_dataToNotify[3] = device_param_getmasterStepSize();
	len = 4*sizeof(uint8_t);

	if(gs_lightingPatternService.conn_handle != BLE_CONN_HANDLE_INVALID)
	{
		//-------------SET the parameters before calling Handle Value Operations
		//	First, reset all the members of params
		memset(&params, 0, sizeof(params));
		//	This attribute has to be notified
		params.type = BLE_GATT_HVX_NOTIFICATION;
		//	Handle of the attribute which has to be notified
		params.handle = gs_lightingPatternService.cube_data_out_char_handles.value_handle;
		//	Pointer to the variable conataing data
		params.p_data = lau8_dataToNotify;
		//	Length of the variable
		params.p_len = &len;
		
		//	This soft device function takes care of notifying the host
		err_code = sd_ble_gatts_hvx(gs_lightingPatternService.conn_handle,&params);
		if(err_code != NRF_SUCCESS)
		{
			return err_code;
		}
	}
	else
		return 32;
	
	return NRF_SUCCESS;
}

/**@brief Function for notifying pattern info. sent every time a command is written
 * @param[in]   rp_patternInfo   	Pattern info
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
static uint32_t lp_notify_pattern_info(uint8_t* rp_patternInfo)
{
	ble_gatts_hvx_params_t params;					//	This will be used to specify the operations to be done
	uint16_t	len = 3*sizeof(uint8_t); 			// 3 bytes are always sent, 0-> pattern, 1->error code 2->master/slave
	uint32_t 	err_code = NRF_SUCCESS;

	if(gs_lightingPatternService.conn_handle != BLE_CONN_HANDLE_INVALID)
	{
		//-------------SET the parameters before calling Handle Value Operations
		//	First, reset all the members of params
		memset(&params, 0, sizeof(params));
		//	This attribute has to be notified
		params.type = BLE_GATT_HVX_NOTIFICATION;
		//	Handle of the attribute which has to be notified
		params.handle = gs_lightingPatternService.pattern_info_char_handles.value_handle;
		//	Pointer to the variable conataing data
		params.p_data = rp_patternInfo;
		//	Length of the variable
		params.p_len = &len;
		
		//	This soft device function takes care of notifying the host
		err_code = sd_ble_gatts_hvx(gs_lightingPatternService.conn_handle,&params);
		if(err_code != NRF_SUCCESS)
		{
			return err_code;
		}
	}
	else
		return 32;
	
	return NRF_SUCCESS;
}

/**@brief Function for notifying Firmware Version info. sent every time a command is written
 * @param[in]   rp_patternInfo   	Pattern info
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
static uint32_t lp_notify_FWVersion_info(uint8_t* rp_patternInfo)
{
	ble_gatts_hvx_params_t params;					//	This will be used to specify the operations to be done
	uint16_t	len = 4*sizeof(uint8_t); 			// 4 bytes are always sent, 0-> FW OPCODE, 1 , 2 and 3 -> FW Vesrion Bytes.
	uint32_t 	err_code = NRF_SUCCESS;

	if(gs_lightingPatternService.conn_handle != BLE_CONN_HANDLE_INVALID)
	{
		//-------------SET the parameters before calling Handle Value Operations
		//	First, reset all the members of params
		memset(&params, 0, sizeof(params));
		//	This attribute has to be notified
		params.type = BLE_GATT_HVX_NOTIFICATION;
		//	Handle of the attribute which has to be notified
		params.handle = gs_lightingPatternService.pattern_info_char_handles.value_handle;
		//	Pointer to the variable conataing data
		params.p_data = rp_patternInfo;
		//	Length of the variable
		params.p_len = &len;
		
		//	This soft device function takes care of notifying the host
		err_code = sd_ble_gatts_hvx(gs_lightingPatternService.conn_handle,&params);
		if(err_code != NRF_SUCCESS)
		{
			return err_code;
		}
	}
	else
		return 32;
	
	return NRF_SUCCESS;
}


/**@brief 	Write handler for cube command attribute
 *
 * @details 
 *
 * @param[in]   rpu8_newState  	Write data
 */
static bool lp_command_write_handler(uint8_t* rpu8_newState)
{
	uint8_t lau8_patternInfo[MAX_PATTERN_INFO_INDEX] = {0};
	uint32_t err_code;
	uint8_t lau8_rgb[lp_set_rgb_max_params];
	uint32_t lu32_numBlinks = 0;
	volatile lp_commands_t le_ReceivedCommand;
	
	//	First byte of the received command provides COMMAND ID
	le_ReceivedCommand = rpu8_newState[0];
	
	// First byte to decide the command received
	switch(le_ReceivedCommand)
	{
		case LP_CMD_SET_LIGHTING_PATTERN:
		{
			
			// Update current pattern and pattern info
			lp_updateLightingPattern(rpu8_newState[1],lau8_patternInfo);
		
			// Store the current pattern in common variable
			ge_currentLightingPattern = (lp_patterns_t)lau8_patternInfo[CURRENT_PATTERN_INDEX];
		
			if(ge_currentLightingPattern == LP_PATTERN_WAVE)
				lu32_numBlinks++;
		}
		break;
	
	
		case LP_CMD_SET_RGB:
		{	
			
			// Update RGB if device is not master
//			if((ge_currentLightingPattern == LP_PATTERN_POUR) && 
//				(device_param_isDeviceMaster() == true))
//			{
//				// do nothing
//			}
//			else	
			{
				//	TO DO:	This command is used in RAINBOW, POUR, 2/3 COLOR BLEND FOR slave devices, pattern. 
				//			App sets the delay based on the speed
				//			Make sure that in the app, delay is >= (connection interval+LED refresh rate). So it should never be below 150 ms
				//	TO DO:	It is assumed that this command is NOT received by the master in Pour, Blend2 / Blend3 patterns. Check on the APP side		
				lau8_rgb[lp_set_rgb_index_red] = rpu8_newState[SETRGB_CMD_INDEX_RED]; 
				lau8_rgb[lp_set_rgb_index_green] = rpu8_newState[SETRGB_CMD_INDEX_GREEN]; 
				lau8_rgb[lp_set_rgb_index_blue] = rpu8_newState[SETRGB_CMD_INDEX_BLUE];
				
				//	TO DO: This is no longer being used by the slave devices as step size is computed purely based on delay. Check
				lau8_rgb[lp_set_rgb_index_stepSize] = (rpu8_newState[SETRGB_CMD_INDEX_STEP_SIZE] == 0) ? 0x0D :rpu8_newState[SETRGB_CMD_INDEX_STEP_SIZE] ;
				
				lau8_rgb[lp_set_rgb_index_delayMs_lsb] = rpu8_newState[SETRGB_CMD_INDEX_DELAY_LSB];
				lau8_rgb[lp_set_rgb_index_delayMs_msb] = rpu8_newState[SETRGB_CMD_INDEX_DELAY_MSB];
				//	Ask for immediate color transition
				lau8_rgb[lp_set_rgb_index_update_immediate] = true;
				// Set RGB color
				lp_setLedColor(lau8_rgb);
			}
			// Update pattern info with current lighting pattern
			lau8_patternInfo[CURRENT_PATTERN_INDEX] = ge_currentLightingPattern;
		}	
		break;
		
		
		case LP_CMD_BLINK_RGB:
		{
			//	TO DO:	This command is used in WAVE, STROBE, KEY patterns and CUBE Identification. 
			//			App sets the delay based on the speed
			//			Make sure that in the app, delay is >= (connection interval+LED refresh rate). So it should never be below 150 ms
			//			
			// activate the LED blink
			lu32_numBlinks = rpu8_newState[BLINK_CMD_INDEX_NUM_OF_BLINKS];	
			
			// Turn off all LEDs without delay before starting the blink
			lau8_rgb[lp_set_rgb_index_red] = lau8_rgb[lp_set_rgb_index_green] = lau8_rgb[lp_set_rgb_index_blue] = 0x00;
			lau8_rgb[lp_set_rgb_index_stepSize] = rpu8_newState[BLINK_CMD_INDEX_STEP_SIZE];
			lau8_rgb[lp_set_rgb_index_delayMs_lsb] = rpu8_newState[BLINK_CMD_INDEX_DELAY_MS_LSB];
			lau8_rgb[lp_set_rgb_index_delayMs_msb] = rpu8_newState[BLINK_CMD_INDEX_DELAY_MS_MSB];
			lp_setLedColor(lau8_rgb);			

		
			while(lu32_numBlinks != 0)
			{
			
				// Turn on all LEDS
				lau8_rgb[lp_set_rgb_index_red] = rpu8_newState[BLINK_CMD_INDEX_RED]; 
				lau8_rgb[lp_set_rgb_index_green] = rpu8_newState[BLINK_CMD_INDEX_GREEN]; 
				lau8_rgb[lp_set_rgb_index_blue] = rpu8_newState[BLINK_CMD_INDEX_BLUE];
				lau8_rgb[lp_set_rgb_index_stepSize] = rpu8_newState[BLINK_CMD_INDEX_STEP_SIZE];
				lau8_rgb[lp_set_rgb_index_delayMs_lsb] = lau8_rgb[lp_set_rgb_index_delayMs_lsb] ;//rpu8_newState[BLINK_CMD_INDEX_DELAY_MS_LSB];
				lau8_rgb[lp_set_rgb_index_delayMs_msb] = lau8_rgb[lp_set_rgb_index_delayMs_msb] ;//rpu8_newState[BLINK_CMD_INDEX_DELAY_MS_MSB];
				lau8_rgb[lp_set_rgb_index_update_immediate] = false;
				lp_setLedColor(lau8_rgb);
			
				// Turn off all LEDS
				lau8_rgb[lp_set_rgb_index_red] = lau8_rgb[lp_set_rgb_index_green] = lau8_rgb[lp_set_rgb_index_blue] = 0x00;
				lau8_rgb[lp_set_rgb_index_stepSize] = rpu8_newState[BLINK_CMD_INDEX_STEP_SIZE];
				lau8_rgb[lp_set_rgb_index_delayMs_lsb] = lau8_rgb[lp_set_rgb_index_delayMs_lsb] ;
				lau8_rgb[lp_set_rgb_index_delayMs_msb] = lau8_rgb[lp_set_rgb_index_delayMs_msb];
				lau8_rgb[lp_set_rgb_index_update_immediate] = false;
				lp_setLedColor(lau8_rgb);		
				
				lu32_numBlinks--;
			
			}
			
			// Update pattern info with current lighting pattern
			lau8_patternInfo[CURRENT_PATTERN_INDEX] = ge_currentLightingPattern;
		}
		break;
		
		case LP_CMD_RESET_CUBE:
		{
			
			// Device is slave
			device_param_setAsSlave();
		
			// deactivate any pattern that is currently running
			CUBE_IO_deactivatePatterns();
		
			// Clear current lighting pattern and update pattern info
			lau8_patternInfo[CURRENT_PATTERN_INDEX] = ge_currentLightingPattern = LP_PATTERN_NONE;
		}
		// reset any parameters that need to be cleared
		break;
		
		case LP_CMD_SET_MASTER:
		{
			if(rpu8_newState[MASTER_FLAG_INDEX])
			{
				// Device is master
				device_param_setAsMaster();
				
				// Set the step size for master
				if(rpu8_newState[STEP_SIZE_INDEX] == 0x00)
				{
					// set default step size if step size is not given
					device_param_setmasterStepSize(DEFAULT_STEP_SIZE);
				}
				else
				{
					device_param_setmasterStepSize(rpu8_newState[MASTER_FLAG_INDEX]);
				}
			}
			else
			{
				// Device is slave
				device_param_setAsSlave();
			}
			// Update pattern info with current lighting pattern
			lau8_patternInfo[CURRENT_PATTERN_INDEX] = ge_currentLightingPattern;			
		}
		break;
		
		case LP_CMD_SET_BASE_COLOR:
		{
			gvu8_color[0]=rpu8_newState[1];
			gvu8_color[1]=rpu8_newState[2];
			gvu8_color[2]=rpu8_newState[3];	
			// Update pattern info with current lighting pattern
			lau8_patternInfo[CURRENT_PATTERN_INDEX] = ge_currentLightingPattern;		
		}
		break;
		
		// Add more commands as needed
		case LP_CMD_FW_VERSION:
		{
			//	TO DO: Why is it returning err_code?
			err_code = lp_notify_FWVersion_info(FW_version_response);
			return err_code;
		}
		break;
		
		case LP_CMD_FOTA_UPDATE:
		{
			
			OTA_ENABLE = rpu8_newState[1];
			//	If CUBE is on charger, do NOT go into DFU mode
			if((CUBE_BAT_IsCUBEOnCharger() )|| (!OTA_ENABLE) )
			{
				lau8_patternInfo[0] = LP_CMD_FOTA_UPDATE;
				lau8_patternInfo[1] = 0;
			}
			else
			{
				lau8_patternInfo[0] = LP_CMD_FOTA_UPDATE;
				lau8_patternInfo[1] = 1;
				//	TO DO: Why is it returning err_code?
				err_code = lp_notify_pattern_info(lau8_patternInfo);
				nrf_delay_ms(1000);
				CUBE_dfu_mode();
			}				
		}
		break;
			
		case LP_CMD_SOUND_ENABLE:
		{
			//	Perform house keeping tasks wheich are otherwise done in the update lighting pattern function
			// deactivate all patterns
			CUBE_IO_deactivatePatterns();
			
			CUBE_IO_setLightingPatternId(LP_PATTERN_AUDIO_RESPONSE);
			
			//	Set flag so that MIC thread becomes active
			Sound_Enable = rpu8_newState[1];
			CUBE_Sound_Pattern = rpu8_newState[2];
			int32_t signal_ena;
			ge_SoundResponseState = SOUND_RESPONSE_ACTIVATE_COMMAND_RECEIVED;
			signal_ena = osSignalSet(CUBE_APP_SoundResponseThread_id, SOUND_RESPONSE_ON_SIGNAL);
			
		}
		break;
		
		case LP_CMD_SOUND_DISABLE:
		{
			Sound_Disable = rpu8_newState[1];
			int32_t signal_ena;
			ge_SoundResponseState = SOUND_RESPONSE_DEACTIVATE_COMMAND_RECEIVED;
			//signal_ena = osSignalSet(CUBE_APP_SoundResponseThread_id, SOUND_RESPONSE_OFF_SIGNAL);
		}
		break;
				
			
		default:
			// INVALID COMMAND
			// Update current pattern to be "INVALID_PATTERN"
			lau8_patternInfo[CURRENT_PATTERN_INDEX] = ge_currentLightingPattern;
		
			// Update error code for ERROR
			lau8_patternInfo[PATTERN_ERR_CODE_INDEX] = 0xFF;
			
			break;
		
	}//	End of switch(le_ReceivedCommand)
	

	// Update if device is master/slave
	lau8_patternInfo[MASTER_STEP_SIZE_INDEX] = device_param_isDeviceMaster();
	
	// Send out notification with pattern info
	err_code = lp_notify_pattern_info(lau8_patternInfo);
	
	return err_code;	
	
}

/**@brief 	Function to update current lighting pattern
 *
 * @details 
 *
 * @param[in]   uint8_t ru8_pattern - lighting pattern id
				uint8_t* rpu8_patternInfo - pattern info to be notified to mobile device
 */
static void lp_updateLightingPattern(uint8_t ru8_pattern,uint8_t* rpu8_patternInfo)
{
	// Update pattern info to reflect the pattern id
	rpu8_patternInfo[CURRENT_PATTERN_INDEX] = ru8_pattern;
	
	// Update error code for no error, error code will be updated below if needed
	rpu8_patternInfo[PATTERN_ERR_CODE_INDEX] = 0;
	
	// Check which lighting pattern is being activated
	switch(ru8_pattern)
	{
		case LP_PATTERN_NONE:
			
			// Device is slave
			device_param_setAsSlave();
		
			// deactivate all patterns
			CUBE_IO_deactivatePatterns();
		
			CUBE_IO_setLightingPatternId(LP_PATTERN_NONE);
				
		
			break;
		
		case LP_PATTERN_POUR:
			
			// Device is slave
			device_param_setAsSlave();
		
			// deactivate any pattern that is currently running
			CUBE_IO_deactivatePatterns();
		
			// activate pour pattern
			CUBE_IO_setLightingPatternId(LP_PATTERN_POUR);
		

			break;
		
		case LP_PATTERN_STROBE:
		case LP_PATTERN_WAVE:			

			// Device is slave
			device_param_setAsSlave();
		
			// deactivate any pattern that is currently running
			CUBE_IO_deactivatePatterns();		
		
		
			// activate wave pattern
			CUBE_IO_setLightingPatternId(LP_PATTERN_WAVE);
			

			break;
		
		case LP_PATTERN_KEYBOARD:
			
			// Device is slave
			device_param_setAsSlave();
		
			// deactivate any pattern that is currently running
			CUBE_IO_deactivatePatterns();
		
			// activate pour pattern
			CUBE_IO_setLightingPatternId(LP_PATTERN_KEYBOARD);
			break;
		
		case LP_PATTERN_RAINBOW:
			// Device is slave
			device_param_setAsSlave();
		
			// deactivate any pattern that is currently running
			CUBE_IO_deactivatePatterns();
		
			// activate pour pattern
			CUBE_IO_setLightingPatternId(LP_PATTERN_RAINBOW);			
			break;
		
		case LP_PATTERN_COLOR_BLEND:
			// Device is slave
			device_param_setAsSlave();
		
			// deactivate any pattern that is currently running
			CUBE_IO_deactivatePatterns();
		
			// activate pour pattern
			CUBE_IO_setLightingPatternId(LP_PATTERN_COLOR_BLEND);			
			break;
		
		case LP_PATTERN_COLOR_BLEND_2:
			// Device is slave
			device_param_setAsSlave();
		
			// deactivate any pattern that is currently running
			CUBE_IO_deactivatePatterns();
		
			// activate pour pattern
			CUBE_IO_setLightingPatternId(LP_PATTERN_COLOR_BLEND_2);			
			break;
			// Add more patterns as needed
		
		case LP_PATTERN_SHAKE:
			
			// Device is slave
			device_param_setAsSlave();
		
			// deactivate any pattern that is currently running
			CUBE_IO_deactivatePatterns();
		
			// activate shake pattern
			CUBE_IO_setLightingPatternId(LP_PATTERN_SHAKE);
			
			break;
		
		default:
			
			// INVALID LIGHTING PATTERN
			// Update pattern info to reflect no pattern change
			rpu8_patternInfo[CURRENT_PATTERN_INDEX] = ge_currentLightingPattern;
		
			// Update error code ERROR
			rpu8_patternInfo[PATTERN_ERR_CODE_INDEX] = 0xFF;
			break;
	}
	
	

}

/**@brief 	Write handler for pattern config attribute
 *
 * @details 
 *
 * @param[in]   rpu8_newState  	Write data
 */
static bool lp_patternConfig_write_handler(uint8_t* rpu8_newState)
{
		
	return true;	
}


/**@brief Function for handling the Connect event.
 *
 * @param[in]   p_cube_cmn       CUBE common Service structure.
 * @param[in]   p_ble_evt   Event received from the BLE stack.
 */
static void lp_on_connect(ble_evt_t * p_ble_evt)
{
	// Update the connection handle 
    gs_lightingPatternService.conn_handle = p_ble_evt->evt.gap_evt.conn_handle;
	
	//	TO DO: Add starting timers / module init events here
	
}


/**@brief Function for handling the Disconnect event.
 *
 * @param[in]   p_cube_cmn      Cube common Service structure.
 * @param[in]   p_ble_evt   Event received from the BLE stack.
 */
static void lp_on_disconnect(ble_evt_t * p_ble_evt)
{
    UNUSED_PARAMETER(p_ble_evt);
	
	// Set the connection handle to be invalid
    gs_lightingPatternService.conn_handle = BLE_CONN_HANDLE_INVALID;
	
	//	TO DO: Things to handle when connection is disabled
	
	// Reset lighting pattern parameters
	// Device is slave
	device_param_setAsSlave();

	// deactivate any pattern that is currently running
	CUBE_IO_deactivatePatterns();

	// Clear current lighting pattern and update pattern info
	ge_currentLightingPattern = LP_PATTERN_NONE;	
}

/**@brief Function for handling the Write event for Lighting pattern service
 *
 * @param[in]   p_ble_evt   Event received from the BLE stack.
 */
static void lp_on_write(ble_evt_t * p_ble_evt)
{
	ble_gatts_evt_write_t * p_evt_write = &p_ble_evt->evt.gatts_evt.params.write;
	bool write_succeeded = false;
	
	/*	WRITE events expected are
			1. CUBE command write
			2. pattern config write
			3. Enable notifications for Data out
	*/

	//	Check if write event is for cube COMMAND
	if(	(p_evt_write->handle == gs_lightingPatternService.cube_command_char_handles.value_handle) && 	// Check if event occured for right char handler
		(p_evt_write->len <= CUBE_PATTERN_ATTR_SIZE_MAX) &&												// Check length of data to write
		(gs_lightingPatternService.mf_cubeCommandWriteHandler != NULL))									// Check if write handler exists
	{
		//	Call the function to handle a write to this attribute
		write_succeeded = gs_lightingPatternService.mf_cubeCommandWriteHandler(p_evt_write->data);

	}
	
	//	Check if write event is for PATTERN CONFIG
	if(	(p_evt_write->handle == gs_lightingPatternService.pattern_config_char_handles.value_handle) &&	// Check if event occured for right char handler
		(p_evt_write->len <= CUBE_PATTERN_ATTR_SIZE_MAX) &&																			// Check length of data to write
		(gs_lightingPatternService.mf_patternConfigWriteHandler != NULL))											// Check if write handler exists
	{
		//	Call the function to handle a write to this attribute
		write_succeeded = gs_lightingPatternService.mf_patternConfigWriteHandler(p_evt_write->data);

	}
	
	//	Write is for Data Out Notification Enable / Disable
	if (p_evt_write->handle == gs_lightingPatternService.cube_data_out_char_handles.cccd_handle)
	{
		// Handle cccd write event for data_out
		lp_cccdWriteDataOutCommand(p_evt_write);
	}
	
	UNUSED_PARAMETER(write_succeeded);
		
}

/**@brief Function for handling write events to the Tilt characteristic.
 *
 * @param[in]   p_cube_lp_pour         	Pour Service structure.
 * @param[in]   p_evt_write   	Write event received from the BLE stack.
 */
static void lp_cccdWriteDataOutCommand(ble_gatts_evt_write_t * p_evt_write)
{
    if (p_evt_write->len == 2)
    {
        // CCCD written, update notification state
		if (ble_srv_is_notification_enabled(p_evt_write->data))
		{
			// Set flag to enable notifications
			device_param_setDataOutEnabled(true);
			
			// Enable IMU
		}
		else
		{
			// Clear flag to disable notifications
			device_param_setDataOutEnabled(false);
			
			// Disable IMU
			
		}

    }
}

/**@brief Function for adding the Cube Data Out characteristic.
 *
 * @param[in]   NONE
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
static uint32_t char_add_cubeDataOut(void)
{
    uint32_t            err_code;
    ble_gatts_char_md_t char_md;
    ble_gatts_attr_t    attr_char_value;
    ble_uuid_t          ble_uuid;
    ble_gatts_attr_md_t attr_md;
	ble_gatts_attr_md_t cccd_md;
	uint8_t lau8_attrInitValue[CUBE_PATTERN_ATTR_SIZE_MAX] = {0};

	//------SET META DATA for Client Configuration Characteristic Descriptor (CCCD)----------	
	memset(&cccd_md, 0, sizeof(cccd_md));					// reset all the meta data prameters
	
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&cccd_md.read_perm);		// the read operation on cccd should be possible without authentication.
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&cccd_md.write_perm); 	// this is used to enable disable notifications
	cccd_md.vloc       = BLE_GATTS_VLOC_STACK;				// This attribute is going to reside in STACK	
	
	//---------------SET META DATA----------------------
	memset(&char_md, 0, sizeof(char_md));	// reset all the meta data parameters
		
	char_md.char_props.read   = 1;			// Can be read
	char_md.char_props.notify = 1 ; 		// Supports notify
	char_md.p_char_user_desc  = NULL;		// No User description string
	char_md.p_char_pf         = NULL;		// No User description string, hence no formatting
	char_md.p_user_desc_md    = NULL;		// No user descriptor meta data
	char_md.p_cccd_md         = &cccd_md; 	// Client Characterisitc Configuration Descriptor
	char_md.p_sccd_md         = NULL;		// No Server Characteristic Configuration Descriptor

	//---------------SET UUID ----------------------
	BLE_UUID_BLE_ASSIGN(ble_uuid, CUBE_DATA_OUT_CHAR);

	//---------------SET META DATA for ATTRIBUTE----------------------
	memset(&attr_md, 0, sizeof(attr_md));	// reset all the parameters of meta data
	
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&attr_md.read_perm);			// can be read
	BLE_GAP_CONN_SEC_MODE_SET_NO_ACCESS(&attr_md.write_perm);	// cannot be written	
	attr_md.vloc       = BLE_GATTS_VLOC_STACK;					//	Attribute is going to reside in STACK	
	
	//	No authentication is ncessary for reading/writing
	attr_md.rd_auth    = 0;
	attr_md.wr_auth    = 0;
	attr_md.vlen       = 0;

	//---------------SET ATTRIBUTE VALUE----------------------	
	memset(&attr_char_value, 0, sizeof(attr_char_value));					//	reset all the parameters of the attribute
	
	attr_char_value.p_uuid    = &ble_uuid;									//	Assign pointer to structure containing UUID of the attribute	
	attr_char_value.p_attr_md = &attr_md;									//	Assign pointer to the structure containing meta data about this attribute	
	attr_char_value.init_len  = CUBE_PATTERN_ATTR_SIZE_MAX*sizeof(uint8_t);	//	Length is given by CUBE_PATTERN_ATTR_SIZE_MAX bytes
	attr_char_value.init_offs = 0;											//	Whole value is used and there are no offsets needed	
	attr_char_value.max_len   = CUBE_PATTERN_ATTR_SIZE_MAX*sizeof(uint8_t);	//	Length is given by CUBE_PATTERN_ATTR_SIZE_MAX bytes	
	attr_char_value.p_value   = lau8_attrInitValue;							//	Initial value is all 0
	
	//	add characteristic to service
	err_code = sd_ble_gatts_characteristic_add(	gs_lightingPatternService.service_handle, 				//	Handle to the service (input)
												&char_md,												//	Characteristic Meta Data
												&attr_char_value,										//	Attribute Value
												&gs_lightingPatternService.cube_data_out_char_handles);	//	Handle to the characteristic assigned from the calling function
																								
    if (err_code != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return err_code;
    }

    return NRF_SUCCESS;
}

/**@brief Function for adding the Cube Command characteristic.
 *
 * @param[in]   NONE
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
static uint32_t char_add_cubeCommand(void)
{
    uint32_t            err_code;
    ble_gatts_char_md_t char_md;
    ble_gatts_attr_t    attr_char_value;
    ble_uuid_t          ble_uuid;
    ble_gatts_attr_md_t attr_md;
	uint8_t lau8_attrInitValue[CUBE_PATTERN_ATTR_SIZE_MAX] = {0};

	//------SET META DATA for Client Configuration Characteristic Descriptor (CCCD)----------	
	// NO CCCD
	
	//---------------SET META DATA----------------------
	memset(&char_md, 0, sizeof(char_md));	// reset all the meta data parameters
		
	char_md.char_props.read   = 1;			// Can be read
	char_md.char_props.write  = 1; 			// can be written
	char_md.p_char_user_desc  = NULL;		// No User description string
	char_md.p_char_pf         = NULL;		// No User description string, hence no formatting
	char_md.p_user_desc_md    = NULL;		// No user descriptor meta data
	char_md.p_cccd_md         = NULL; 		// No Client Characterisitc Configuration Descriptor
	char_md.p_sccd_md         = NULL;		// No Server Characteristic Configuration Descriptor

	//---------------SET UUID ----------------------
	BLE_UUID_BLE_ASSIGN(ble_uuid, CUBE_COMMAND_CHAR);

	//---------------SET META DATA for ATTRIBUTE----------------------
	memset(&attr_md, 0, sizeof(attr_md));	// reset all the parameters of meta data
	
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&attr_md.read_perm);			// can be read
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&attr_md.write_perm);		// can be written	
	attr_md.vloc       = BLE_GATTS_VLOC_STACK;					//	Attribute is going to reside in STACK	
	
	//	No authentication is ncessary for reading/writing
	attr_md.rd_auth    = 0;
	attr_md.wr_auth    = 0;
	attr_md.vlen       = 0;

	//---------------SET ATTRIBUTE VALUE----------------------	
	memset(&attr_char_value, 0, sizeof(attr_char_value));					//	reset all the parameters of the attribute
	
	attr_char_value.p_uuid    = &ble_uuid;									//	Assign pointer to structure containing UUID of the attribute	
	attr_char_value.p_attr_md = &attr_md;									//	Assign pointer to the structure containing meta data about this attribute	
	attr_char_value.init_len  = CUBE_PATTERN_ATTR_SIZE_MAX*sizeof(uint8_t);	//	Length is given by CUBE_PATTERN_ATTR_SIZE_MAX bytes
	attr_char_value.init_offs = 0;											//	Whole value is used and there are no offsets needed	
	attr_char_value.max_len   = CUBE_PATTERN_ATTR_SIZE_MAX*sizeof(uint8_t);	//	Length is given by CUBE_PATTERN_ATTR_SIZE_MAX bytes	
	attr_char_value.p_value   = lau8_attrInitValue;							//	Initial value is all 0
	
	//	add characteristic to service
	err_code = sd_ble_gatts_characteristic_add(	gs_lightingPatternService.service_handle, 				//	Handle to the service (input)
												&char_md,												//	Characteristic Meta Data
												&attr_char_value,										//	Attribute Value
												&gs_lightingPatternService.cube_command_char_handles);	//	Handle to the characteristic assigned from the calling function
																								
    if (err_code != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return err_code;
    }

    return NRF_SUCCESS;
}


/**@brief Function for adding the pattern info characteristic.
 *
 * @param[in]   NONE
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
static uint32_t char_add_patternInfo(void)
{
    uint32_t            err_code;
    ble_gatts_char_md_t char_md;
    ble_gatts_attr_t    attr_char_value;
    ble_uuid_t          ble_uuid;
    ble_gatts_attr_md_t attr_md;
	ble_gatts_attr_md_t cccd_md;
	uint8_t lau8_attrInitValue[CUBE_PATTERN_ATTR_SIZE_MAX] = {0};

	//------SET META DATA for Client Configuration Characteristic Descriptor (CCCD)----------	
	memset(&cccd_md, 0, sizeof(cccd_md));					// reset all the meta data prameters
	
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&cccd_md.read_perm);		// the read operation on cccd should be possible without authentication.
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&cccd_md.write_perm); 	// This is used to enable/disable notifications
	cccd_md.vloc       = BLE_GATTS_VLOC_STACK;				// This attribute is going to reside in STACK	
	
	//---------------SET META DATA----------------------
	memset(&char_md, 0, sizeof(char_md));	// reset all the meta data parameters
		
	char_md.char_props.read   = 1;			// Can be read
	char_md.char_props.notify  = 1; 		// can be notified
	char_md.p_char_user_desc  = NULL;		// No User description string
	char_md.p_char_pf         = NULL;		// No User description string, hence no formatting
	char_md.p_user_desc_md    = NULL;		// No user descriptor meta data
	char_md.p_cccd_md         = &cccd_md; 	// Client Characterisitc Configuration Descriptor
	char_md.p_sccd_md         = NULL;		// No Server Characteristic Configuration Descriptor

	//---------------SET UUID ----------------------
	BLE_UUID_BLE_ASSIGN(ble_uuid, CUBE_PATTERN_INFO_CHAR);

	//---------------SET META DATA for ATTRIBUTE----------------------
	memset(&attr_md, 0, sizeof(attr_md));	// reset all the parameters of meta data
	
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&attr_md.read_perm);			// can be read
	BLE_GAP_CONN_SEC_MODE_SET_NO_ACCESS(&attr_md.write_perm); 	// cannot be written	
	attr_md.vloc       = BLE_GATTS_VLOC_STACK;					//	Attribute is going to reside in STACK	
	
	//	No authentication is ncessary for reading/writing
	attr_md.rd_auth    = 0;
	attr_md.wr_auth    = 0;
	attr_md.vlen       = 0;

	//---------------SET ATTRIBUTE VALUE----------------------	
	memset(&attr_char_value, 0, sizeof(attr_char_value));					//	reset all the parameters of the attribute
	
	attr_char_value.p_uuid    = &ble_uuid;									//	Assign pointer to structure containing UUID of the attribute	
	attr_char_value.p_attr_md = &attr_md;									//	Assign pointer to the structure containing meta data about this attribute	
	attr_char_value.init_len  = CUBE_PATTERN_ATTR_SIZE_MAX*sizeof(uint8_t);	//	Length is given by CUBE_PATTERN_ATTR_SIZE_MAX bytes
	attr_char_value.init_offs = 0;											//	Whole value is used and there are no offsets needed	
	attr_char_value.max_len   = CUBE_PATTERN_ATTR_SIZE_MAX*sizeof(uint8_t);	//	Length is given by CUBE_PATTERN_ATTR_SIZE_MAX bytes	
	attr_char_value.p_value   = lau8_attrInitValue;							//	Initial value is all 0
	
	//	add characteristic to service
	err_code = sd_ble_gatts_characteristic_add(	gs_lightingPatternService.service_handle, 				//	Handle to the service (input)
												&char_md,												//	Characteristic Meta Data
												&attr_char_value,										//	Attribute Value
												&gs_lightingPatternService.pattern_info_char_handles);	//	Handle to the characteristic assigned from the calling function
																								
    if (err_code != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return err_code;
    }

    return NRF_SUCCESS;
}

#if 0
/**@brief Function for adding the pattern info characteristic.
 *
 * @param[in]   NONE
 *
 * @return      NRF_SUCCESS on success, otherwise an error code.
 */
static uint32_t char_add_patternConfig(void)
{
    uint32_t            err_code;
    ble_gatts_char_md_t char_md;
    ble_gatts_attr_t    attr_char_value;
    ble_uuid_t          ble_uuid;
    ble_gatts_attr_md_t attr_md;
	uint8_t lau8_attrInitValue[CUBE_PATTERN_ATTR_SIZE_MAX] = {0};

	//------SET META DATA for Client Configuration Characteristic Descriptor (CCCD)----------	
	// NO CCCD
	
	//---------------SET META DATA----------------------
	memset(&char_md, 0, sizeof(char_md));	// reset all the meta data parameters
		
	char_md.char_props.read   = 1;			// Can be read
	char_md.char_props.write  = 1; 			// can be written
	char_md.p_char_user_desc  = NULL;		// No User description string
	char_md.p_char_pf         = NULL;		// No User description string, hence no formatting
	char_md.p_user_desc_md    = NULL;		// No user descriptor meta data
	char_md.p_cccd_md         = NULL; 		// No Client Characterisitc Configuration Descriptor
	char_md.p_sccd_md         = NULL;		// No Server Characteristic Configuration Descriptor

	//---------------SET UUID ----------------------
	BLE_UUID_BLE_ASSIGN(ble_uuid, CUBE_PATTERN_CONFIG_CHAR);

	//---------------SET META DATA for ATTRIBUTE----------------------
	memset(&attr_md, 0, sizeof(attr_md));	// reset all the parameters of meta data
	
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&attr_md.read_perm);			// can be read
	BLE_GAP_CONN_SEC_MODE_SET_OPEN(&attr_md.write_perm);		// can be written	
	attr_md.vloc       = BLE_GATTS_VLOC_STACK;					//	Attribute is going to reside in STACK	
	
	//	No authentication is ncessary for reading/writing
	attr_md.rd_auth    = 0;
	attr_md.wr_auth    = 0;
	attr_md.vlen       = 0;

	//---------------SET ATTRIBUTE VALUE----------------------	
	memset(&attr_char_value, 0, sizeof(attr_char_value));					//	reset all the parameters of the attribute
	
	attr_char_value.p_uuid    = &ble_uuid;									//	Assign pointer to structure containing UUID of the attribute	
	attr_char_value.p_attr_md = &attr_md;									//	Assign pointer to the structure containing meta data about this attribute	
	attr_char_value.init_len  = CUBE_PATTERN_ATTR_SIZE_MAX*sizeof(uint8_t);	//	Length is given by CUBE_PATTERN_ATTR_SIZE_MAX bytes
	attr_char_value.init_offs = 0;											//	Whole value is used and there are no offsets needed	
	attr_char_value.max_len   = CUBE_PATTERN_ATTR_SIZE_MAX*sizeof(uint8_t);	//	Length is given by CUBE_PATTERN_ATTR_SIZE_MAX bytes	
	attr_char_value.p_value   = lau8_attrInitValue;							//	Initial value is all 0
	
	//	add characteristic to service
	err_code = sd_ble_gatts_characteristic_add(	gs_lightingPatternService.service_handle, 				//	Handle to the service (input)
												&char_md,												//	Characteristic Meta Data
												&attr_char_value,										//	Attribute Value
												&gs_lightingPatternService.pattern_config_char_handles);	//	Handle to the characteristic assigned from the calling function
																								
    if (err_code != NRF_SUCCESS)
    {
		//	If there was a problem doing this task, exit with error code
        return err_code;
    }

    return NRF_SUCCESS;
}
#endif




