/*
$DESCRIPTION		: Timers for CUBE

$Copyright			: CRATUS TECHNOLOGY INC, 2015-16 

$Project			  : CUBE

$Author				  : Dhruv

*/

/*! 
	\file	CUBE_timers.c
 *
 * @defgroup CUBE_timers	WIQB Timer Module
 * @{
 * @ingroup CUBE_timers	WIQB Timer Module
 * @brief File contains all timer related functions 
 */
 
/* INCLUDE FILES */
#include <stdint.h>
#include <stdbool.h>
#include "softdevice_handler.h"
#include "app_timer.h"
#include "nrf_drv_timer.h"
#include "nrf_timer.h"
#include "nrf_drv_timer.h"
#include "app_util_platform.h"
#include "app_scheduler.h"
#include "nrf_delay.h"
#include "CUBE_ble_cmn.h"
#include "CUBE_BLE_lightingPattern.h"
#include "CUBE_timers.h"
#include "CUBE_IO_LED.h"
#include "CUBE_IO_ACCL.h"
#include "CUBE_IO_MIC.h"
#include "CUBE_APP_LightingController.h"
#include "cmsis_os.h"
#include "CUBE_BLE_Reliable_Write.h"
#include "CUBE_IO_Motor.h"
#include "CUBE_IO_LIS2DH.h"
#include "nrf_drv_ppi.h"
#include "nrf_adc.h"
#include "arm_math.h"
#include "CUBE_APP_SoundResponse.h"

/* MODULE EXTERNAL DATA DEFINITIONS        *ddddddd*/
extern cube_io_raw_accl_t raw_accl_values;			// Global Accelerometer buffer defined in CUBE_IO_ACCL.c file
volatile uint32_t gu32_100msTimeCounter = 0;
extern  SoundResponseState_t ge_SoundResponseState;

/* MODULE INTERNAL CONSTANT DEFINITIONS    *ccccccc*/
#define	BLE_NOTIFY_TIMER_INTERVAL		100					/**< Time in ms to send BLE notifications */
#define ACCEL_SAMPLE_TIMER_INTERVAL		20					/**< Time in ms to sample Accelerometer data */
#define	HAPTIC_DRIVE_TIME				60					/* Duration for which haptic feedback is active. It is HAPTIC_DRIVE_TIME*LED_REFRESH_RATE */

/* MODULE INTERNAL TYPE DEFINITIONS        *ttttttt*/

/* MODULE INTERNAL MACRO DEFINITIONS       *mmmmmmm*/

/* MODULE INTERNAL FUNCTION PROTOTYPES     *fffffff*/

/** Functions for BLE Notifications Timer **/


/** Functions for LED Timer **/
/*!
    \brief 		Function to create timer for LED fading
*/


bool CUBE_BAT_monitor(void);

/* MODULE INTERNAL DATA DEFINITIONS        *ddddddd*/

/******************************************* Timers used in the cube Firmware are converted as Threads ******************************/



osThreadId CUBE_IO_LED_thread_id;
osThreadId CUBE_IO_blenotify_thread_id;
osThreadId CUBE_IO_accl_thread_id;
osThreadId CUBE_APP_SoundResponseThread_id;


void CUBE_IO_LED_thread(void const * arg);  
osThreadDef(CUBE_IO_LED_thread, osPriorityBelowNormal, 1, 0); 

void CUBE_IO_blenotify_thread(void const * arg);  
osThreadDef(CUBE_IO_blenotify_thread, osPriorityBelowNormal, 1, 0);

void CUBE_IO_accl_thread(void const * arg);  
osThreadDef(CUBE_IO_accl_thread, osPriorityBelowNormal, 1, 0);

void CUBE_APP_SoundResponseThread(void const * arg);  
osThreadDef(CUBE_APP_SoundResponseThread, osPriorityBelowNormal, 1, 0);

bool CUBE_APP_ProcessAccelDataForShakePattern(void);

osMessageQDef(LED_MsgBox, 24, RGB_LED);               // Define message queue
osMessageQId  LED_MsgBox;



/** Functions for the Hardware TIMER2 **/
/*!
    \brief 		The intention to create this timer is to 
*/


/* EXPORTED (GLOBAL or External) FUNCTIONS *ggggggg*/
/*!
    \fn 		CUBE_APP_ThreadsInit()
	\brief 		Function to initialize all timers used in CUBE.

	\details 	Function initializes the app timer module. It creates a timer for LED control
				and a timer for periodic BLE notifications
		
    \param[in] 	NONE        
					
    \return	    NONE 	

    \note		This function only creates the timers, the timers are not started here. Timers are started after initializations

*/
void CUBE_APP_ThreadsInit(void)
{
	osStatus CUBE_APP_SoundResponseThread_stat;
	
	// create msg queue. This is needed for information exchange between SoundResponse and LED control thread
	LED_MsgBox = osMessageCreate(osMessageQ(LED_MsgBox), NULL);  
	
	//	LED Controller thread
	CUBE_IO_LED_thread_id = osThreadCreate(osThread(CUBE_IO_LED_thread), NULL);
	if(CUBE_IO_LED_thread_id  == NULL)
		NVIC_SystemReset();
	UNUSED_VARIABLE(CUBE_IO_LED_thread_id);
	
	//	This thread is useful only during Pour, 2 Color Mix and 3 Color Mix patterns
	//	It populates LED queue with R,G,B data corresponding to accelerometer tilt values
	CUBE_IO_blenotify_thread_id = osThreadCreate(osThread(CUBE_IO_blenotify_thread), NULL);
	if(CUBE_IO_blenotify_thread_id  == NULL)
		NVIC_SystemReset();
	UNUSED_VARIABLE(CUBE_IO_blenotify_thread_id);

	//	This thread continuously reads accelerometer values. Ideally, it should be turned on only when needed.
	//	TO DO: Scope for optimization
	CUBE_IO_accl_thread_id = osThreadCreate(osThread(CUBE_IO_accl_thread), NULL);
	if(CUBE_IO_accl_thread_id   == NULL)
		NVIC_SystemReset();

	UNUSED_VARIABLE(CUBE_IO_accl_thread_id);
	
	//	Sound Response thread: Will sleep until the command for turning on Audio from APP is received
	CUBE_APP_SoundResponseThread_id = osThreadCreate(osThread(CUBE_APP_SoundResponseThread), NULL);
	if(CUBE_APP_SoundResponseThread_id    == NULL)
		NVIC_SystemReset();
	
	UNUSED_VARIABLE(CUBE_APP_SoundResponseThread_id);

}



extern volatile uint32_t  gu32_Connected;
extern volatile uint8_t Sound_Enable;
volatile uint32_t gu32_HapticOnCount=0;
extern volatile bool gb_driveHaptic;

//	TO DO: This thread can be totally removed and its functionality can be part of accl_Thread.
//			To make sure that BLE notification does NOT happen at the same rate as that of reading the accelerometer data
void CUBE_IO_blenotify_thread(void const * arg)
{
	while(1)
	{
		CUBE_APP_RunLightingController(NULL, 0);
		// Send Data Out characteristic BLE notifications
		lp_notify_dataOut();
		UNUSED_VARIABLE(osDelay(100));
	}
}

void CUBE_IO_accl_thread(void const * arg)
{

	uint32_t lu32_AccelerometerRange=0;
	
	
	while(1)
	{
		
		//	TO DO: Turn this thread ON only for the patterns that need accelerometer data.
		//			POUR, BLEND 2, BLEND 3 and SHAKE
		
		int16_t l16_acclRawValues[3];
		uint8_t lu8_index;
		if(device_param_isDeviceMaster())
		{
			lu8_index = getPattern();
			if(lu8_index == LP_PATTERN_POUR || lu8_index == LP_PATTERN_COLOR_BLEND || lu8_index == LP_PATTERN_COLOR_BLEND_2)
			{
				//	Make sure that the range is set appropriately
				if(lu32_AccelerometerRange != LIS2DH_FULLSCALE_2)
				{
					//	Set it to full scale
					LIS2DH_selectScale(LIS2DH_FULLSCALE_2);
					lu32_AccelerometerRange = LIS2DH_FULLSCALE_2;
					UNUSED_VARIABLE(osDelay(100));
				}
				
				//	TO DO: Ensure that Range is +/- 2g as it may be set to +/- 16g due to shake pattern
				//	12-bit resolution
				CUBE_IO_ACCL_control(ACCL_READ_ALL_AXIS,l16_acclRawValues);
				raw_accl_values.ms16_acclRawXaxisValues = l16_acclRawValues[0];
				raw_accl_values.ms16_acclRawYaxisValues = l16_acclRawValues[1];
				raw_accl_values.ms16_acclRawZaxisValues = l16_acclRawValues[2];
				
				//	Computes moving average for a given buffer size	
				CUBE_APP_ProcessAccelerometerData();
			}
			else if(LP_PATTERN_SHAKE == lu8_index)
			{
				if(lu32_AccelerometerRange != LIS2DH_FULLSCALE_16)
				{
					//	Set it to full scale
					LIS2DH_selectScale(LIS2DH_FULLSCALE_16);
					lu32_AccelerometerRange = LIS2DH_FULLSCALE_16;
					UNUSED_VARIABLE(osDelay(100));
				}
				
				//	If haptic motor is vibrating, don't process accelerometer data
				if(!gb_driveHaptic)
				{					
					//	TO DO: Set sensitivity to +/- 16g for shake pattern
					CUBE_IO_ACCL_control(ACCL_READ_ALL_AXIS,l16_acclRawValues);
					raw_accl_values.ms16_acclRawXaxisValues = l16_acclRawValues[0];
					raw_accl_values.ms16_acclRawYaxisValues = l16_acclRawValues[1];
					raw_accl_values.ms16_acclRawZaxisValues = l16_acclRawValues[2];
					
					CUBE_APP_ProcessAccelDataForShakePattern();
					osDelay(30);
					if(gb_driveHaptic)
						osDelay(100);
				}
			} 
			
		}
		//	Sampling at 50 Hz
		UNUSED_VARIABLE(osDelay(20));
		
	}//	End of while(1)
	
}
#define		DEBUG_BUFFER_SIZE			100
uint32_t gau32_DebugBuffer[DEBUG_BUFFER_SIZE];
uint32_t index_count = 0;

extern volatile bool gu32_SoundResponseIsActive;
void CUBE_IO_LED_thread(void const * arg)
{
	uint8_t  lau8_rgb[lp_set_rgb_max_params];
	uint32_t lu32_RefreshRate=0;
	osEvent  evt;
	RGB_LED* rptr_color_Index;

	while(1)
	{
		//	Data on which LED controller needs to act and refresh rates are different
		//	if the active pattern is Sound Response. For all other patterns, a constant refresh rate is used
		if(gu32_SoundResponseIsActive == true)
		{
			evt = osMessageGet(LED_MsgBox, osWaitForever);  // wait for message
			if (evt.status == osEventMessage) 
			{
				//NVIC_SystemReset();
				rptr_color_Index = evt.value.p;
				lau8_rgb[lp_set_rgb_index_red] = rptr_color_Index->red ;                
				lau8_rgb[lp_set_rgb_index_green] =  rptr_color_Index->green  ;           
				lau8_rgb[lp_set_rgb_index_blue] =  rptr_color_Index->blue ;              
				lau8_rgb[lp_set_rgb_index_stepSize] = 0 ;         
				lau8_rgb[lp_set_rgb_index_delayMs_lsb] =  0;        
				lau8_rgb[lp_set_rgb_index_delayMs_msb] =   0;
				//	Ask for immediate color transition
				lau8_rgb[lp_set_rgb_index_update_immediate] = true;

				// Set RGB color
				lp_setLedColor(lau8_rgb);

			//osPoolFree(mpool, rptr_color_Index);
				
			}//	End of evt.status....
		
			lu32_RefreshRate = REFREASH_RATE_FOR_SOUND_RESPONSE;
		}
		else
		{
			lu32_RefreshRate = LED_TIMER_INTERVAL;
		}

		//	This is not an ideal way to do it. But works.
		//	If haptic feedback is on, provide haptic feedback
		if(gb_driveHaptic)
		{
			
			gu32_HapticOnCount++;
			//	Turn On/Off haptic @ of LED refresh rate
			CUBE_IO_MOTR_Control(CUBE_IO_MOTR_SET_PWM,50);
			//	If duration of haptic feedback is over, reset the flag to stop giving haptic feedback
			if(gu32_HapticOnCount >= HAPTIC_DRIVE_TIME)	
			{
				gb_driveHaptic = false;
				gu32_HapticOnCount = 0;
				CUBE_IO_MOTR_Control(CUBE_IO_MOTR_SET_PWM,0);
			}
				
		}
		else
			gu32_HapticOnCount = 0;
		
		//	If the CUBE is disconnected state, get all the LEDs to OFF state			
		if(gu32_Connected == 0)
		{
			lau8_rgb[lp_set_rgb_index_red] = 0 ;                
			lau8_rgb[lp_set_rgb_index_green] = 0;           
			lau8_rgb[lp_set_rgb_index_blue] =  0;              
			lau8_rgb[lp_set_rgb_index_stepSize] = 0 ;         
			lau8_rgb[lp_set_rgb_index_delayMs_lsb] =  0;        
			lau8_rgb[lp_set_rgb_index_delayMs_msb] =   0;
			//	Ask for immediate color transition
			lau8_rgb[lp_set_rgb_index_update_immediate] = true;

			// Set RGB color
			lp_setLedColor(lau8_rgb);
		}
		
		CUBE_IO_runLedPattern();
		
		//	Delay based on refresh rate			
		osDelay(lu32_RefreshRate);
			
	}
}



/*!
    \fn			CUBE_APP_EnterSoundResponseThreadConfig()
	\brief 		Terminate the threads that are not needed during Sound Response
		
    \param[in] 	NONE        
					
    \return	    NONE 	

	\note 		

*/
void CUBE_APP_EnterSoundResponseThreadConfig()
{
	osStatus thread_stat;
	//	Terminate BLE Notify Thread. This thread updates the LED Thread with commands recived from the App
	thread_stat = osThreadTerminate(CUBE_IO_blenotify_thread_id);
	//	This thread continuously reads the data from accelerometer
	thread_stat = osThreadTerminate(CUBE_IO_accl_thread_id);

}

/*!
    \fn			CUBE_APP_ExitSoundResponseThreadConfig()
	\brief 		Terminate the threads that are not needed during Sound Response
		
    \param[in] 	NONE        
					
    \return	    NONE 	

	\note 		

*/
void CUBE_APP_ExitSoundResponseThreadConfig()
{
	osStatus ls_ThreadStat;
	
	//	Terminate BLE Notify Thread. This thread updates the LED Thread with commands recived from the App
	CUBE_IO_blenotify_thread_id = osThreadCreate(osThread(CUBE_IO_blenotify_thread), NULL);
	UNUSED_VARIABLE(CUBE_IO_blenotify_thread_id);

	//	This thread continuously reads the data from accelerometer
	CUBE_IO_accl_thread_id = osThreadCreate(osThread(CUBE_IO_accl_thread), NULL);
	UNUSED_VARIABLE(CUBE_IO_accl_thread_id);
	
}

/*!
    \fn			Cube_timers_start()
	\brief 		Function to start all timers and to enable interrupt for MPU 
		
    \param[in] 	NONE        
					
    \return	    NONE 	

	\note 		MPU interrupt can be moved to a separate file/function so that all interrupts can be enabled from a single location
				and it is not part of the timer code

*/

/* 			INTERNAL FUNCTIONS *ggggggg*/

