/*
 *  cratustech.com - Copyright (C) CRATUS Technology, Inc. 2015-16
 * 
 *	Redistribution and use in source and binary forms, with or without modification,
 * 	are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of CRATUS Technology, Inc nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * 	THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * 	AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * 	IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * 	DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * 	FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * 	DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * 	SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * 	CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * 	OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * 	OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */
 
/** \file CUBE_APP_LightingController.c
 *
 * \brief Cube App Lighting Controller related services are implemented here
 */
 
/* INCLUDE FILES */
#include <string.h>
#include "arm_math.h"
#include "CUBE_APP_LightingController.h"
#include "nordic_common.h"
#include "CUBE_IO_LED.h"
#include "CUBE_IO_ACCL.h"
#include "CUBE_IO_MOTOR.h"

#define ACCEL_BUFFER_SIZE			8					// Max size 

/* MODULE EXTERNAL DATA DEFINITIONS        *ddddddd*/
extern volatile bool gb_driveHaptic;
extern cube_io_raw_accl_t raw_accl_values;			// Global Accelerometer buffer defined in CUBE_IO_ACCL.c file
extern volatile uint8_t gvu8_color[3];
extern volatile uint32_t gu32_100msTimeCounter;

/* MODULE INTERNAL CONSTANT DEFINITIONS    *ccccccc*/

/* MODULE INTERNAL DATA DEFINITIONS        *ddddddd*/
static uint8_t rgb[3] = {0};
static int16_t accelBuffX[16] = {0};
static int16_t accelBuffY[16] = {0};
static int16_t accelBuffZ[16] = {0};
static int16_t filteredAccelX = 0;
static int16_t filteredAccelY = 0;
static int16_t filteredAccelZ = 0;
static int32_t accelAverageX = 0;
static int32_t accelAverageY = 0;
static int32_t accelAverageZ = 0;
static bool filledCircularBuffer = false;

static volatile uint32_t su32_motorTimeCounter = 0;	

/* MODULE INTERNAL MACRO DEFINITIONS       *mmmmmmm*/

bool CUBE_APP_LightingControllerInit(void)
{
		// Nothing to init
		return true;
}

bool CUBE_APP_ProcessAccelerometerData(void)
{
		static uint8_t lu8_bufferIndex = 0;
		//int16_t ls16_acclRawValue[3];
		
		// Get data from Accelerometer global buffer   --> Considered Resolution +/-2g for Accelerometer
		accelBuffX[lu8_bufferIndex] = raw_accl_values.ms16_acclRawXaxisValues;
		accelBuffY[lu8_bufferIndex] = raw_accl_values.ms16_acclRawYaxisValues;
		accelBuffZ[lu8_bufferIndex] = raw_accl_values.ms16_acclRawZaxisValues;
	
		accelAverageX += accelBuffX[lu8_bufferIndex];
		accelAverageY += accelBuffY[lu8_bufferIndex];
		accelAverageZ += accelBuffZ[lu8_bufferIndex];
	
		lu8_bufferIndex++;
	
		if(lu8_bufferIndex > (ACCEL_BUFFER_SIZE-1) && !filledCircularBuffer)
		{	
				filledCircularBuffer = true;
		}
		
		lu8_bufferIndex &= (ACCEL_BUFFER_SIZE-1);
	
		filteredAccelX = accelAverageX >> 5;
		filteredAccelY = accelAverageY >> 5;
		filteredAccelZ = accelAverageZ >> 5;
		
		if(filledCircularBuffer)
		{
				accelAverageX -= accelBuffX[lu8_bufferIndex];
				accelAverageY -= accelBuffY[lu8_bufferIndex];
				accelAverageZ -= accelBuffZ[lu8_bufferIndex];
		}
		
		return true;
}

static bool getLEfromFilteredData(void)
{
	// Based on different Accelerometer axis data decide RGB color value
	if(fabs(filteredAccelX) > 5)		// Red
	{
		rgb[0] = fabs(filteredAccelX);		
		if(rgb[0]>255)
			rgb[0] = 255;
	}
	else
	{
		rgb[0] = 0;
	}

	if(filteredAccelY > 5)					// Green
	{ 
		rgb[1] = fabs(filteredAccelY); 			
		if(rgb[1]>255)
			rgb[1] = 255;
	}
	else
	{
		rgb[1] = 0;
	}

	if(filteredAccelY < -5)					// Blue
	{
		rgb[2] = fabs(filteredAccelY); 			
		if(rgb[2]>255)
			rgb[2] = 255;
	}
	else
	{
		rgb[2] = 0;
	}

	return true;
}

#define		THRESHOLD_VALUE 		0xA00000			//	Corresponds to .... 

bool CUBE_APP_ProcessAccelDataForShakePattern(void)
{
		int16_t accel_data[3];
		static int16_t si16_avgAccel[3];	         //stores a moving average on x,y,z samples
		uint16_t lu16_deltaAccel[3];	             //relative difference between moving average and current sample
		uint8_t lau8_rgb[lp_set_rgb_max_params] = {0};

		uint16_t lu16_ss;
		static uint16_t su16_prev_ss;
		int32_t ls32_totalMeasuredEnergy=0;
		
		accel_data[0] = raw_accl_values.ms16_acclRawXaxisValues;
		accel_data[1] = raw_accl_values.ms16_acclRawYaxisValues;
		accel_data[2] = raw_accl_values.ms16_acclRawZaxisValues;
		
#if 0		
		//delta ( moving avg Vs current sample)
		lu16_deltaAccel[0] = fabs(si16_avgAccel[0] - accel_data[0]);
		lu16_deltaAccel[1] = fabs(si16_avgAccel[1] - accel_data[1]);
		lu16_deltaAccel[2] = fabs(si16_avgAccel[2] - accel_data[2]);
		
		// moving average over signed acceleration 
		si16_avgAccel[0] = (int16_t)( si16_avgAccel[0]*0.75f + accel_data[0]*0.25f );	//75% dominated by prev samples
		si16_avgAccel[1] = (int16_t)( si16_avgAccel[1]*0.75f + accel_data[1]*0.25f );	
		si16_avgAccel[2] = (int16_t)( si16_avgAccel[2]*0.75f + accel_data[2]*0.25f );	
		
		lu16_ss=MAX(lu16_deltaAccel[0],lu16_deltaAccel[1]);
		lu16_ss=MAX(lu16_deltaAccel[2],lu16_ss);


		// Notifications enabled, update LED
		if(lu16_ss<600)
		{
			lau8_rgb[lp_set_rgb_index_green] = 0;
			lau8_rgb[lp_set_rgb_index_red] = 0;
			lau8_rgb[lp_set_rgb_index_blue] = 0;	//rgb[2];	
			//gb_driveHaptic = false;
		}
		else
		{
			if(lu16_ss > 2572)
			{
					//	Turn on the haptic for 2 seconds
					gb_driveHaptic = true;
					su32_motorTimeCounter = gu32_100msTimeCounter;
				
					// Clear Previous Accelerometer data
					si16_avgAccel[0] = 0;
					si16_avgAccel[1] = 0;
					si16_avgAccel[2] = 0;
				
					// Clear LED State
					lau8_rgb[lp_set_rgb_index_red] = 0;
					lau8_rgb[lp_set_rgb_index_green] = 0;	 	
					lau8_rgb[lp_set_rgb_index_blue] = 0;	
			}
			else
			{
					if(gvu8_color[0])
					{
						lau8_rgb[lp_set_rgb_index_red] = (uint8_t)(lu16_ss>>4);
						lau8_rgb[lp_set_rgb_index_green] = 0;	 	
						lau8_rgb[lp_set_rgb_index_blue] = 0;	
					}
					else if(gvu8_color[1])
					{
						lau8_rgb[lp_set_rgb_index_red] = 0;
						lau8_rgb[lp_set_rgb_index_green] = (uint8_t)(lu16_ss>>4);	
						lau8_rgb[lp_set_rgb_index_blue] = 0;	
					}
					else if(gvu8_color[2])
					{
						lau8_rgb[lp_set_rgb_index_red] = 0;
						lau8_rgb[lp_set_rgb_index_green] = 0;
						lau8_rgb[lp_set_rgb_index_blue] = (uint8_t)(lu16_ss>>4);	
					}			
			}				
		}		
		

		//lau8_rgb[lp_set_rgb_index_stepSize] = 8;//device_param_getmasterStepSize();
		lau8_rgb[lp_set_rgb_index_delayMs_lsb] = 0x46;	// 500 ms time	
		lau8_rgb[lp_set_rgb_index_delayMs_msb] = 0x0;			
		lp_setLedColor(lau8_rgb);	
		
		su16_prev_ss= lu16_ss;			
#else
		//	It is assumed that the multiplication is signed as the data types are all signed
		ls32_totalMeasuredEnergy = 	accel_data[0] * accel_data[0] + 
									accel_data[1] * accel_data[1] +
									accel_data[2] * accel_data[2];

		//	values are 12-bit in resolution
		if(ls32_totalMeasuredEnergy >= THRESHOLD_VALUE)
		{
			gb_driveHaptic = true;
			ls32_totalMeasuredEnergy = 0x2FC000;
		}
		
		if(gvu8_color[0])
		{
			lau8_rgb[lp_set_rgb_index_red] = (uint8_t)(ls32_totalMeasuredEnergy>>14);
			lau8_rgb[lp_set_rgb_index_green] = 0;	 	
			lau8_rgb[lp_set_rgb_index_blue] = 0;	
		}
		else if(gvu8_color[1])
		{
			lau8_rgb[lp_set_rgb_index_red] = 0;
			lau8_rgb[lp_set_rgb_index_green] = (uint8_t)(ls32_totalMeasuredEnergy>>14);	
			lau8_rgb[lp_set_rgb_index_blue] = 0;	
		}
		else if(gvu8_color[2])
		{
			lau8_rgb[lp_set_rgb_index_red] = 0;
			lau8_rgb[lp_set_rgb_index_green] = 0;
			lau8_rgb[lp_set_rgb_index_blue] = (uint8_t)(ls32_totalMeasuredEnergy>>14);	
		}			
		
		lau8_rgb[lp_set_rgb_index_delayMs_lsb] = 0;	// 500 ms time	
		lau8_rgb[lp_set_rgb_index_delayMs_msb] = 0;			
		lp_setLedColor(lau8_rgb);	
		
#endif
		
		
	return true;
}

extern 	uint32_t volatile gu32_Connected;

void CUBE_APP_RunLightingController(void * p_event_data, uint16_t event_size)
{

		uint8_t lau8_rgb[lp_set_rgb_max_params] = {0};

		
		switch(getPattern())	// Get pattern Info
		{			
		
			/*-------------------------------------------------------------------------------------------------------------
																		POUR LIGHTING PATTERN
			---------------------------------------------------------------------------------------------------------------*/
			case LP_PATTERN_POUR:
			case LP_PATTERN_COLOR_BLEND:
			case LP_PATTERN_COLOR_BLEND_2:
			{
				if(device_param_isDeviceMaster() == true)
				{
						//CUBE_APP_ProcessAccelerometerData();
						
						getLEfromFilteredData();
						// Notifications enabled, update LED
						lau8_rgb[lp_set_rgb_index_red] = rgb[0]; 
						lau8_rgb[lp_set_rgb_index_green] = rgb[1]; 
						lau8_rgb[lp_set_rgb_index_blue] = rgb[2];
						lau8_rgb[lp_set_rgb_index_delayMs_lsb] = 0x2C;				// 100 ms Delay
						lau8_rgb[lp_set_rgb_index_delayMs_msb] = 0x1;		
						lau8_rgb[lp_set_rgb_index_update_immediate] = false;			// Don't want immediate color change -> want smooth change with proper steps
						lp_setLedColor(lau8_rgb);		
				}
				break;
			}
			
			/*-------------------------------------------------------------------------------------------------------------
																		SHAKE LIGHTING PATTERN
			---------------------------------------------------------------------------------------------------------------*/
			case LP_PATTERN_SHAKE:
			{
					uint32_t lu32_motorTimeDifference = 0;
					
					//	Processing is done in Accelerometer Thread
					
					// If Haptic get disabled then only process Accelerometer data
//					if(!gb_driveHaptic)
//					{
//						CUBE_APP_ProcessAccelDataForShakePattern();
//					}
//					else
//					{	
//						// After 2 second from enabled time, stop Haptic Motor driver
//						lu32_motorTimeDifference = gu32_100msTimeCounter - su32_motorTimeCounter;
//						if(lu32_motorTimeDifference>20)
//						{
//								gb_driveHaptic = false;
//								CUBE_IO_MOTR_Control(CUBE_IO_MOTR_SET_PWM,0);
//						}
//						else
//						{
//							uint32_t lu32_dutyCycle = 0;


//							lu32_dutyCycle = 50 * (lu32_motorTimeDifference & 0x01);//lu32_motorTimeDifference<<1;
//							//	Change the intensity of vibration 
//							CUBE_IO_MOTR_Control(CUBE_IO_MOTR_SET_PWM,(1+lu32_dutyCycle));
//						}
//					}
					break;
			}
			
			case LP_PATTERN_STROBE:
			case LP_PATTERN_WAVE:
				break;
			
			case LP_PATTERN_RAINBOW:
				break;
			
			case LP_PATTERN_KEYBOARD:
				break;
			
			case LP_PATTERN_NONE:
				break;
			
			case LP_PATTERN_MAX:
				break;
			
		}	// End of switch case
}
